# 機能設計書 22-注意機構（Attention Mechanism）

## 概要

本ドキュメントは、TensorFlow/Kerasにおける注意機構（Attention Mechanism）の機能設計を記述する。スケーリングドットプロダクトアテンション（Luong-style）とアディティブアテンション（Bahdanau-style）の2つの主要な注意機構レイヤーを提供する。

### 本機能の処理概要

**業務上の目的・背景**：注意機構は、入力シーケンスの全位置に対して重要度の重みを動的に計算することで、モデルが「どの部分に注目すべきか」を学習する仕組みである。これにより、長いシーケンスでの情報損失を防ぎ、機械翻訳・質問応答・テキスト要約などの精度を大幅に向上させる。Transformer架構の基盤技術でもある。

**機能の利用シーン**：Seq2Seq翻訳モデルのデコーダでエンコーダ出力への注意を計算する場合、テキスト類似性の計算、マルチモーダル情報の融合、自己注意（Self-Attention）を使ったシーケンス内部の依存関係モデリングなどで使用される。

**主要な処理内容**：
1. Query・Key・Valueテンソルの受け取りと検証
2. Queryと Keyの類似度スコア計算（ドット積またはアディティブ方式）
3. Causalマスク・Valueマスクの適用
4. Softmaxによるアテンション分布の計算
5. ドロップアウトの適用（学習時）
6. アテンション分布とValueの加重和による出力計算

**関連システム・外部連携**：リカレント層（No.21）やコア層（No.24）のDense層と組み合わせて使用されることが多い。

**権限による制御**：特になし。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| - | - | - | 画面機能マッピングに該当なし |

## 機能種別

計算処理（テンソル演算・注意重み計算）

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| inputs | list[Tensor] | Yes | [query, value] or [query, value, key] | リスト長2または3 |
| mask | list[Tensor] | No | [query_mask, value_mask] | リスト長2またはinputsと同じ |
| training | bool | No | 学習モード（ドロップアウト適用）か | デフォルト: None |
| return_attention_scores | bool | No | アテンションスコアも返すか | デフォルト: False |
| use_scale | bool | No | スケーリング変数を使用するか（Attention） | デフォルト: False |
| causal | bool | No | Causalマスクを適用するか | デフォルト: False |
| dropout | float | No | ドロップアウト率 | 0.0-1.0 |

### 入力データソース

Kerasモデルの前段レイヤー出力。queryは[batch_size, Tq, dim]、valueは[batch_size, Tv, dim]形状。

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| output | Tensor | 注意出力 [batch_size, Tq, dim] |
| attention_scores | Tensor | (オプション) [batch_size, Tq, Tv] |

### 出力先

次段のKerasレイヤーまたはモデル出力。

## 処理フロー

### 処理シーケンス

```
1. 入力検証（_validate_call_args）
   └─ inputsがリスト形式で長さ2-3か検証
2. Query, Value, Keyの分離
   └─ keyが未指定の場合、valueをkeyとして使用
3. スコア計算（_calculate_scores）
   └─ Attention: scores = matmul(query, key^T) * scale
   └─ AdditiveAttention: scores = reduce_sum(tanh(query + key))
4. マスク構築
   └─ value_maskの拡張 [batch, 1, Tv]
   └─ causal=Trueの場合、下三角マスク生成
5. スコア適用（_apply_scores）
   └─ マスク位置に大きな負値を加算
   └─ softmaxでアテンション分布計算
   └─ 学習時はドロップアウト適用
   └─ matmul(distribution, value)
6. クエリマスク適用
   └─ q_mask位置の出力をゼロ化
```

### フローチャート

```mermaid
flowchart TD
    A[inputs: query, value, key] --> B[入力検証]
    B --> C[スコア計算]
    C --> D{causal?}
    D -->|Yes| E[下三角マスク生成]
    D -->|No| F[マスクなし]
    E --> G[value_maskとマージ]
    F --> G
    G --> H[softmax適用]
    H --> I{training?}
    I -->|Yes| J[ドロップアウト適用]
    I -->|No| K[そのまま]
    J --> L[matmul: weights * value]
    K --> L
    L --> M[query_mask適用]
    M --> N[出力返却]
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-22-01 | Key省略時の動作 | keyが未指定の場合valueがkeyとしても使用される | callメソッド呼び出し時 |
| BR-22-02 | Causalマスク | causal=Trueでposition i はposition j>i を参照不可 | デコーダの自己注意 |
| BR-22-03 | マスクのパディング | マスク位置のスコアに-1e9（float32）or -65504（float16）を加算 | マスク適用時 |

### 計算ロジック

- **Attention (Luong-style)**: `scores = Q * K^T / scale`, `output = softmax(scores) * V`
- **AdditiveAttention (Bahdanau-style)**: `scores = V_s * tanh(W_q * Q + W_k * K)`, `output = softmax(scores) * V`

## データベース操作仕様

該当なし（インメモリテンソル演算のみ）

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| ValueError | 入力検証エラー | inputsがリスト形式でない | リスト [query, value] を渡す |
| ValueError | 入力検証エラー | inputsの長さが2未満または3超 | 長さ2または3のリストにする |
| ValueError | マスク検証エラー | maskがリスト形式でない | リスト [query_mask, value_mask] を渡す |

### リトライ仕様

該当なし。

## トランザクション仕様

該当なし。

## パフォーマンス要件

- float16使用時のパディング値は65504（float16最大値）に設定される
- ドロップアウトは学習時のみ適用され推論時のオーバーヘッドはない

## セキュリティ考慮事項

特になし。

## 備考

- 本実装はDense/CNNネットワーク向けであり、RNNネットワーク向けではない（docstringに明記）
- Transformer全体の実装にはMultiHeadAttention等の別クラスが必要

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

Query/Key/Valueの3テンソル構造とマスクの仕組みを把握する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | dense_attention.py | `tensorflow/python/keras/layers/dense_attention.py` | BaseDenseAttentionのdocstring（33-73行目）でQ/K/V構造を理解 |

**読解のコツ**: Attention論文（Vaswani et al., 2017）のFigure 2の用語に準拠している。queryが「問い合わせ」、keyが「索引」、valueが「値」。

#### Step 2: ベースクラスの共通処理

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | dense_attention.py | `tensorflow/python/keras/layers/dense_attention.py` | BaseDenseAttention.call（141-177行目）の処理フロー |
| 2-2 | dense_attention.py | `tensorflow/python/keras/layers/dense_attention.py` | _apply_scores（94-138行目）のマスク適用とsoftmax計算 |

**主要処理フロー**:
1. **141-149行目**: inputs分解（q, v, k）、keyがない場合はvalueで代替
2. **152行目**: _calculate_scoresでスコア計算
3. **156-167行目**: causalマスクの生成（下三角行列）
4. **168-170行目**: マスクのマージとスコア適用
5. **121-128行目**: パディングマスク位置に大きな負値を加算
6. **131行目**: nn.softmaxでアテンション分布計算
7. **136-137行目**: 学習時のドロップアウト適用

#### Step 3: Attention（Luong-style）実装

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | dense_attention.py | `tensorflow/python/keras/layers/dense_attention.py` | Attentionクラス（219-353行目） |

**主要処理フロー**:
- **336-348行目**: _calculate_scoresでmatmul(query, key, transpose_b=True)、use_scale時はscale変数を乗算

#### Step 4: AdditiveAttention（Bahdanau-style）実装

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 4-1 | dense_attention.py | `tensorflow/python/keras/layers/dense_attention.py` | AdditiveAttentionクラス（356行目以降） |

### プログラム呼び出し階層図

```
tf.keras.layers.Attention / AdditiveAttention
    │
    ├─ BaseDenseAttention (base class)
    │      ├─ call()
    │      │    ├─ _validate_call_args()
    │      │    ├─ _calculate_scores(query, key)  [サブクラスで実装]
    │      │    ├─ _merge_masks(v_mask, causal_mask)
    │      │    ├─ _apply_scores(scores, value, scores_mask)
    │      │    │    ├─ nn.softmax(scores)
    │      │    │    └─ nn.dropout(weights)  [学習時のみ]
    │      │    └─ query_mask適用
    │      └─ compute_mask()
    │
    ├─ Attention._calculate_scores()
    │    └─ math_ops.matmul(query, key, transpose_b=True)
    │
    └─ AdditiveAttention._calculate_scores()
         └─ reduce_sum(tanh(query + key), axis=-1)
```

### データフロー図

```
[入力]                       [処理]                          [出力]

query [B,Tq,D]   ───▶  _calculate_scores           ───▶  output [B,Tq,D]
key   [B,Tv,D]   ───▶      │
value [B,Tv,D]   ───▶  scores [B,Tq,Tv]                  attention_scores
                            │                              [B,Tq,Tv] (optional)
v_mask [B,Tv]    ───▶  mask & softmax
q_mask [B,Tq]    ───▶  attention_distribution
                            │
                        matmul(dist, value)
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| dense_attention.py | `tensorflow/python/keras/layers/dense_attention.py` | ソース | BaseDenseAttention/Attention/AdditiveAttentionの定義 |
| base_layer.py | `tensorflow/python/keras/engine/base_layer.py` | ソース | Layerベースクラス |
| nn.py | `tensorflow/python/ops/nn.py` | ソース | softmax, dropout関数 |
| math_ops.py | `tensorflow/python/ops/math_ops.py` | ソース | matmul等のテンソル演算 |
